O2_Model <- function(b, k, Model, alpha, JzMax, data.source, year, O2_threshold = O2_threshold){
  O2.consumption.rate <- O2.decay(b = b, k = k, Model = Model, alpha = alpha, JzMax = JzMax)
  nday = 231 #Numbers of days to simulate
  dbeg = 135 #First day of simulated stratification
  
  #Create empty matrix with the good dimensions
  O2.modelled.matrix = as.data.frame(matrix(nrow=nrow(O2.consumption.rate), ncol = nday))
  #Set first profile at 12.3 mg O2/L
  O2.modelled.matrix[,1] = 12.3
  
  #Model daily profiles based on consumption rate
  for(i in 2:ncol(O2.modelled.matrix)){
    O2.modelled.matrix[,i] = 12.3 - O2.consumption.rate[,2]*(i-1)
  }
  #Change negative values to 0
  O2.modelled.matrix[O2.modelled.matrix<0] = 0
  
  #Anaerobic duration threshold
  O2.threshold = O2_threshold
  
  #Change values below threshold to 1, others to 0
  O2.modelled.matrix.bin <- ifelse(O2.modelled.matrix <= O2.threshold, 1, 0)
  
  #Create new object to calculate anaerobic duration
  O2.modelled.AD <- O2.modelled.matrix.bin
  for(j in 2:ncol(O2.modelled.AD)){
    temp = O2.modelled.AD[,j-1]+O2.modelled.AD[,j]
    temp = ifelse(temp > O2.modelled.AD[,j-1], temp, 0)
    O2.modelled.AD[,j] = temp
  }
  
  #Add a column for depths and colnames
  O2.modelled.AD = cbind(c(30:47,47.5,48,48.5), O2.modelled.AD)
  colnames(O2.modelled.AD)[1] = "Depth"
  colnames(O2.modelled.AD)[2:(nday+1)] = paste(seq(dbeg,dbeg+nday-1,1))
  
  #Transform into dataframe
  O2.modelled.AD = as.data.frame(O2.modelled.AD)
  
  #Transform to long format
  O2.modelled.AD.long = pivot_longer(O2.modelled.AD,
                                            cols = 2:ncol(O2.modelled.AD),
                                            names_to = "DOY", values_to = "AD")
  O2.modelled.AD.long = as.data.frame(O2.modelled.AD.long)
  
  #See results in ugly graph
  if(Model=="Livingstone") Model.gg = "Linear"
  if(Model=="Log") Model.gg = "Log-linear"
  if(Model=="Exp") Model.gg = "Exponential-plateau"
  print(ggplot(O2.modelled.AD.long[O2.modelled.AD.long$Depth!=47.5 & O2.modelled.AD.long$Depth!=48.5,])+
          geom_raster(aes(x=as.numeric(DOY), y=Depth, fill=AD))+
          scale_y_reverse()+
          scale_fill_gradient(low="white", high="red")+
          ggtitle(paste0(year,", ", Model.gg, " model"), "- Progression of anoxia, excluding lake mixing")+
          xlab("DOY")+
          labs(fill = bquote(AD~(.(O2.threshold)~mg~L^-1))))
  
  
  
  #Align the modeled oxygen profile with the real data to associate anoxic age with 
  #reduced compounds concentration using the lowest sum of squared difference
  
  if(data.source == "YSI") {
    year.i = grep(year, YSI.list)
    Ar.YSI = read.csv(paste0("./",YSI.list[year.i]))
  }  
  
  #Remove NA at bottom of profiles with niminum value of the profile
  for(j in 1:ncol(Ar.YSI))
  {
    if(any(is.na(Ar.YSI[,j]))) Ar.YSI[is.na(Ar.YSI[,j]),j] = min(Ar.YSI[,j], na.rm=T)
  }
  
  #Transform to long format
  Ar.long = pivot_longer(Ar.YSI, cols = 2:ncol(Ar.YSI), names_to = "Date", values_to = "DO_mgL")
  #Split Date into Year, months and day
  Ar.long$Date = substring(Ar.long$Date, 2,20) %>% strptime("%Y.%m.%d.%H.%M.%S")
  
  Ar.long$DOY = as.numeric(strftime(Ar.long$Date, format = "%j"))
  
  Ar.long = Ar.long %>% mutate(Year = substring(Date, 1, 4),
                               mm = substring(Date, 6, 7),
                               dd = substring(Date, 9, 10))
  
  #Keep only values below 30 meters to match other methods
  Ar.long.deep = Ar.long[Ar.long$Depth_m>=30,]
  
  #Order the dataframe
  Ar.long.deep = Ar.long.deep[order(Ar.long.deep$Depth_m ,Ar.long.deep$DOY),]
  
  
  #Select the date matching chemistry observations
  #In 2020, Arendsee was sampled on 2020-05-26 (DOY 147) and 2020-08-25	
  Ar.first.profile = Ar.long.deep[,-2] %>% filter(mm == "05", dd == "26") %>% 
    group_by(Depth_m) %>% summarize(DO_mgL = mean(DO_mgL))
  
  # Ar.second.profile = Ar.long.deep[,-2] %>% filter(mm == "08", dd == "25") %>% 
  #   group_by(Depth_m) %>% summarize(DO_mgL = mean(DO_mgL))
  
  if(year==2017) Match.temp.1 = O2.modelled.matrix[-c(1,2),] else Match.temp.1 = O2.modelled.matrix[-c(1,2,3),]
  
  
  for(i in 1:ncol(O2.modelled.matrix)){
    Match.temp.1[,i] = O2.modelled.matrix[,i] - Ar.first.profile[,2]
  }
  
  Match.temp.1.abs = (Match.temp.1)^2
  Match.day.1 = which(colSums(Match.temp.1.abs) == min(colSums(Match.temp.1.abs)))
  Match.day.1.RMSE = sqrt(sum(Match.temp.1.abs[Match.day.1])/length(Match.temp.1.abs[Match.day.1]))
  
  O2.modelled.AD = O2.modelled.AD[,-c(2:(Match.day.1-1))]
  colnames(O2.modelled.AD)[c(2:ncol(O2.modelled.AD))] = seq(147,ncol(O2.modelled.AD)+145,1)
  output = list()
  output[[1]] = O2.modelled.AD
  output[[2]] = Match.day.1
  output[[3]] = Match.day.1.RMSE
  return(output)
}